import React from 'react';
import styled from 'styled-components';
import { InterDataType } from '@/api/interface';
import {
  GetCompanyInfoById,
  GetCompanyInspectionById,
} from '@/api/interface/service';
import { ServiceAPI } from '@/api/modules/service';
import BreadcrumbView from '@/components/breadcrumb';
import LayoutView from '@/components/layout';
import ServiceHeadView from '@/components/service-detail/comp/detail-head';
import ServiceStoreView from '@/components/service-detail/comp/detail-store';
import { wrapper } from '@/store';
import { setGlobalData } from '@/store/module/globalData';

// 商品详情类型
type DetailType = InterDataType<GetCompanyInspectionById>;
// 商城详情类型
type StoreType = InterDataType<GetCompanyInfoById>;

// 使用wrapper.getServerSideProps高阶函数包裹getServerSideProps函数
export const getServerSideProps = wrapper.getServerSideProps(
  (store) => async (context) => {
    // 商品id
    const id: number = Number(context.params?.id);
    // 商品详情
    let serviceDetail: DetailType | undefined;
    // 店铺详情
    let storeDetail: StoreType | undefined;
    // 获取服务详情
    const getCompanyInspection = async () => {
      const res = await ServiceAPI.getCompanyInspectionById({ id });
      if (res && res.code === '200') {
        serviceDetail = res.result;
        // console.log('获取服务详情 --->', serviceDetail);
      }
    };
    // 获取店铺详情
    const getCompanyInfo = async () => {
      const res = await ServiceAPI.getCompanyInfoById({
        id: Number(serviceDetail?.companyInfoId),
      });
      if (res && res.code === '200') {
        storeDetail = res.result;
      }
    };
    // 获取价格单位列表
    const getPriceUnitList = async () => {
      const res = await ServiceAPI.listInspectionPriceUnit();
      if (res && res.code === '200') {
        store.dispatch(setGlobalData({ priceUnitList: res?.result || [] }));
      }
    };
    // 依次获取接口数据
    await (async () => {
      await getCompanyInspection();
      await getCompanyInfo();
      await getPriceUnitList();
    })();
    return { props: { serviceDetail, storeDetail } };
  },
);

const ServiceDetailView: React.FC<{
  id: number;
  serviceDetail: DetailType;
  storeDetail: StoreType;
}> = ({ serviceDetail, storeDetail }) => {
  return (
    <LayoutView>
      <ServiceDetailWrap>
        <BreadcrumbView />
        <div className="flex-start align-start">
          <ServiceHeadView detail={serviceDetail} />
          <ServiceStoreView detail={serviceDetail} store={storeDetail} />
        </div>
        <div className="product-title">商品详情</div>
        <div className="product-richText">
          {serviceDetail?.detailPage && (
            <div
              className="content-html"
              dangerouslySetInnerHTML={{ __html: serviceDetail?.detailPage }}
            />
          )}
        </div>
      </ServiceDetailWrap>
    </LayoutView>
  );
};

export default ServiceDetailView;
// 样式
const ServiceDetailWrap = styled.div`
  position: relative;
  max-width: 1190px;
  box-sizing: border-box;
  padding: 2rem 0 0 0;
  margin: 0 auto;
  .product-title {
    position: relative;
    width: 100%;
    text-align: center;
    padding: 1.5rem 0;
    color: #989898;
    font-size: 14px;
    &::after,
    &::before {
      position: absolute;
      content: '';
      top: 50%;
      left: 40%;
      width: 4.74rem;
      height: 0.02rem;
      background: #d4d4d4;
    }
    &::before {
      left: auto;
      right: 40%;
    }
  }
  .product-richText {
    position: relative;
    width: 100%;
    margin-bottom: 3.5rem;
    .content-html {
      width: 100%;
      img {
        width: 100%;
      }
    }
  }
`;
